%-------------------------------------------------------------------------%
% Copyright (c) 2017 Modenese L.                                          %
%                                                                         %
% Licensed under the Apache License, Version 2.0 (the "License");         %
% you may not use this file except in compliance with the License.        %
% You may obtain a copy of the License at                                 %
% http://www.apache.org/licenses/LICENSE-2.0.                             %
%                                                                         %
% Unless required by applicable law or agreed to in writing, software     %
% distributed under the License is distributed on an "AS IS" BASIS,       %
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or         %
% implied. See the License for the specific language governing            %
% permissions and limitations under the License.                          %
%                                                                         %
%    Author: Luca Modenese, October 2017                                  %
%    email:  l.modenese08@imperial.ac.uk                                  %
%-------------------------------------------------------------------------%
% Example script to preprocess marker data using the Biomechanical Tool Kit
% (BTK). Marker trajectories stored in c3d files is extracted, rotated
% in order for the global axis to match OpenSim conventions and saved as
% trc file. 
%
% Please note that in trc files time vector starts from zero.
%-------------------------------------------------------------------------%
clear; clc;

%---------------- USER'S SETTINGS ------------------
% folder where to look for c3d files
folder_c3d = '..\c3d_files';
% folder where to store the trc files
folder_trc = '..\trc_files';
% axis pointing upwards when acquisition is visualized in MOKKA
up = [0 0 1];
% axis pointing forward when acuisition is visualized in MOKKA
forward = [-1 0 0];
%---------------------------------------------------

% check if trc folder exists, if it doesn't create it
if ~isdir(folder_trc);    mkdir(folder_trc);  end

% Rotation matrix from MOCAP to ISB/OPenSim
RotMat_MOCAP2ISB = [forward;   up;  cross(forward, up)]';

% getting list of c3d from the specified folder
list_of_c3d = dir([folder_c3d,filesep,'*.c3d']);
% number of trials N
N = size(list_of_c3d,1); 

if isempty(list_of_c3d)==1
    error(['There are no c3d files in specified folder: ',folder_c3d]);
end
% looping through all the c3d files found in folder
% NB: it is assumed that the transformation to apply is the same!
for n = 1:N
    
    % get c3d file name
    c3d_file_name = list_of_c3d(n).name;
    
    % inform user of what is happening.
    disp(['Processing trial ',num2str(n),'/',num2str(N),': ', c3d_file_name,'.'])
    
    % c3d file with path
    c3d_file = fullfile(folder_c3d, c3d_file_name);
    
    %trc file name with path
    [~,name,~] = fileparts(c3d_file);
    trc_file = fullfile(folder_trc,[name,'.trc']); 
    
    % reading acquisition
    acq = btkReadAcquisition(c3d_file);
    
    % rotating marker data to ISB/OpenSim convention.
    acq_ISB = rotateMarkerData(acq, RotMat_MOCAP2ISB);
    
    % print the trc files
    btkWriteAcquisition(acq_ISB, trc_file);
    
    %delete cloned acquisition
    btkDeleteAcquisition(acq_ISB);
    
    % delete acquisition
    btkDeleteAcquisition(acq);
end

% inform user of end of processing
disp('All trials were processed.');